/*
 * Linux driver for Eagletron trackerpod:  http://www.trackercam.com
 * Steve Hill, cheesy@sackheads.org
 *
 * Derived from dabusb and pwc drivers, with help from the Rubini book.
 * This is my first attempt at a device driver, hopefully it's not
 * too bad!
 *
 * Note: This driver does not seem to work with uhci.o
 */

#include <linux/kernel.h>
#include <linux/usb.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/poll.h>

#include "trackerpod-ioctl.h"

#define TRACKERPOD_NAME		"trackerpod"
#define TRACKERPOD_DEVNAME	"tpod%d"
#define TRACKERPOD_MINOR	96
#define TRACKERPOD_MAX_DEVS	16

#define TRACKERPOD_VEND		0x3125
#define TRACKERPOD_PROD		0x1

#define TRACKERPOD_AXIS_PAN	0x60
#define TRACKERPOD_AXIS_TILT	0x40

#define	TRACKERPOD_MIN_PAN	0x00
#define TRACKERPOD_MAX_PAN	0xFF

#define	TRACKERPOD_MIN_TILT	0x16
#define TRACKERPOD_MAX_TILT	0xEA

#define TRACKERPOD_EOC_MAX_URBS	104

#ifndef CLAMP
#define CLAMP(x, low, high)\
	(((x) > (high)) ? (high) : (((x) < (low)) ? (low) : (x)))
#endif

#define trackerpod_error(msg, args...)\
	printk(KERN_ERR     "%s: " msg "\n" , pdev->name , ## args)

#define trackerpod_warn(msg, args...)\
	printk(KERN_WARNING "%s: " msg "\n" , pdev->name , ## args)

#define trackerpod_msg(msg, args...)\
	printk(KERN_INFO    "%s: " msg "\n" , pdev->name , ## args)

#ifdef TRACKERPOD_DEBUG
#define trackerpod_debug(msg, args...)\
	printk(KERN_DEBUG   "%s: " msg "\n" , pdev->name , ## args)
#else
#define trackerpod_debug(msg, args...)
#endif

#ifdef TRACKERPOD_DEBUG_PKTS
#define trackerpod_debug_pkt(msg, args...)\
	printk(KERN_DEBUG   "%s: " msg "\n" , pdev->name , ## args)
#else
#define trackerpod_debug_pkt(msg, args...)
#endif

MODULE_LICENSE("GPL");

static struct usb_driver trackerpod_usb_ops;

static struct usb_device_id trackerpod_id_table[] =
{
  { USB_DEVICE(TRACKERPOD_VEND, TRACKERPOD_PROD) },
  { },
};

MODULE_DEVICE_TABLE(usb, trackerpod_id_table);

struct trackerpod_device
{
  char name[sizeof(TRACKERPOD_DEVNAME)];
  struct usb_device *udev;
  wait_queue_head_t remove_ok;
  struct semaphore lock;
  int initialised;
  int unplugged;
  int open;
  int minor;
  int shutdown_on_close;
  __u8 eoc[8];
  __u8 usb_buffer[8];
  int num_urbs;
  int pan;
  int tilt;
};

void trackerpod_populate_struct(struct trackerpod_device *pdev, int minor)
{
  init_waitqueue_head(&pdev->remove_ok);
  init_MUTEX(&pdev->lock);

  pdev->minor = minor;
  sprintf(pdev->name, TRACKERPOD_DEVNAME, minor);

  pdev->initialised = 0;
  pdev->unplugged   = 0;
  pdev->open        = 0;
  pdev->num_urbs    = 0;
  pdev->pan         = 0x80;
  pdev->tilt        = 0x80;
  pdev->shutdown_on_close = 0;
  memcpy(pdev->eoc, "\x09\xc8\x09\xc8\x09\xc8\x09\xc8", 8);
}

int trackerpod_regenerate_eoc(struct trackerpod_device *pdev);

int trackerpod_send(struct trackerpod_device *pdev, __u8 *buffer)
{
  int i, nsent = 0, rc;

  trackerpod_debug("trackerpod_send called");

  if (buffer == NULL)
  {
    trackerpod_debug_pkt("Send EOC");
    memcpy(pdev->usb_buffer, pdev->eoc, 8);
  }
  else
  {
    trackerpod_debug_pkt("Send (pre XOR): "\
		    "%02x %02x %02x %02x %02x %02x %02x %02x",
		    buffer[0],
		    buffer[1],
		    buffer[2],
		    buffer[3],
		    buffer[4],
		    buffer[5],
		    buffer[6],
		    buffer[7]);

    for (i = 0; i < 8; i++)
    {
      pdev->usb_buffer[i] = buffer[i] ^ pdev->eoc[i];
    }
  }

  trackerpod_debug_pkt("Send URB %d (postXOR): "\
		  "%02x %02x %02x %02x %02x %02x %02x %02x",
		  pdev->num_urbs,
		  pdev->usb_buffer[0],
		  pdev->usb_buffer[1],
		  pdev->usb_buffer[2],
		  pdev->usb_buffer[3],
		  pdev->usb_buffer[4],
		  pdev->usb_buffer[5],
		  pdev->usb_buffer[6],
		  pdev->usb_buffer[7]);


  rc = usb_bulk_msg(pdev->udev, usb_sndbulkpipe(pdev->udev, 1),
		  pdev->usb_buffer, 8, &nsent, 5 * HZ);

  trackerpod_debug("trackerpod_send rc = %d, nsent = %d", rc, nsent);

  if (rc == 0 && nsent < 8)
  {
    trackerpod_error("Short send!");
    return -EIO;
  }

  if (rc == 0 || nsent == 8)
  {
    pdev->num_urbs++;
  }

  return rc;
}

int trackerpod_send_multiple(struct trackerpod_device *pdev,
		__u8 *buffer, int n)
{
  int rc = 0;

  if (pdev->num_urbs > TRACKERPOD_EOC_MAX_URBS)
  {
    rc = trackerpod_regenerate_eoc(pdev);
  }

  while (rc == 0 && n-- > 0)
  {
    rc = trackerpod_send(pdev, buffer);
  }

  return rc;
}

int trackerpod_recv(struct trackerpod_device *pdev)
{
  int i, nrecv = 0, rc;

  trackerpod_debug("trackerpod_recv called");

  rc = usb_bulk_msg(pdev->udev, usb_rcvbulkpipe(pdev->udev, 1),
		  pdev->usb_buffer, 8, &nrecv, 5 * HZ);

  if (rc == 0 && nrecv < 8)
  {
    trackerpod_error("Short recv!");
    return -EIO;
  }

  if (rc == 0 && nrecv == 8)
  {
    trackerpod_debug_pkt("Received (pre XOR): "\
		  "%02x %02x %02x %02x %02x %02x %02x %02x",
		  pdev->usb_buffer[0],
		  pdev->usb_buffer[1],
		  pdev->usb_buffer[2],
		  pdev->usb_buffer[3],
		  pdev->usb_buffer[4],
		  pdev->usb_buffer[5],
		  pdev->usb_buffer[6],
		  pdev->usb_buffer[7]);

    for (i = 0; i < 8; i++)
    {
      pdev->usb_buffer[i] = pdev->usb_buffer[i] ^ pdev->eoc[i];
    }

    trackerpod_debug_pkt("Received (post XOR): "\
		  "%02x %02x %02x %02x %02x %02x %02x %02x",
		  pdev->usb_buffer[0],
		  pdev->usb_buffer[1],
		  pdev->usb_buffer[2],
		  pdev->usb_buffer[3],
		  pdev->usb_buffer[4],
		  pdev->usb_buffer[5],
		  pdev->usb_buffer[6],
		  pdev->usb_buffer[7]);
  }

  trackerpod_debug("trackerpod_recv rc = %d, nrecv = %d", rc, nrecv);

  return rc;
}

int trackerpod_regenerate_eoc(struct trackerpod_device *pdev)
{
  int old_urbcount, rc = 0;

  trackerpod_debug("regenerate_eoc called");

  /* Stop send_multiple calling us again! */
  old_urbcount = pdev->num_urbs;
  pdev->num_urbs = 0;

  rc = (rc != 0) ? rc : trackerpod_send_multiple(pdev,
		 		 "\x81\x00\x01\xc0\x01\xc0\x01\xc0", 2);
  rc = (rc != 0) ? rc : trackerpod_recv(pdev);

  if (rc == 0)
  {
    /* Check the pan/tilt sent to us by the pod */

    if ((pdev->usb_buffer[2] ^ 0x41) != pdev->pan)
    {
      trackerpod_warn("Pod tells me pan is %02x, expecting %02x",
		      pdev->usb_buffer[2] ^ 0x41, pdev->pan);
      pdev->pan = pdev->usb_buffer[2] ^ 0x41;
    }

    if ((pdev->usb_buffer[4] ^ 0x41) != pdev->tilt)
    {
      trackerpod_warn("Pod tells me tilt is %02x, expecting %02x",
		      pdev->usb_buffer[4] ^ 0x41, pdev->tilt);
      pdev->tilt = pdev->usb_buffer[4] ^ 0x41;
    }

    /* Now actually calculate new EOC sequence */
    pdev->eoc[0] = pdev->eoc[2] = pdev->eoc[4] = pdev->eoc[6]
	     = pdev->usb_buffer[0];

    pdev->eoc[3] = pdev->eoc[5] = pdev->eoc[7]
	     = pdev->usb_buffer[0] ^ 0xC1;

    pdev->eoc[1] = pdev->eoc[3] ^ ((pdev->eoc[3] & 0x0F) << 4) ^ 0x80;

    /* Dont count the URBs sent for negotiation */
    pdev->num_urbs = 0;

    trackerpod_debug_pkt("New eoc: %02x %02x %02x %02x %02x %02x %02x %02x",
		  pdev->eoc[0],
		  pdev->eoc[1],
		  pdev->eoc[2],
		  pdev->eoc[3],
		  pdev->eoc[4],
		  pdev->eoc[5],
		  pdev->eoc[6],
		  pdev->eoc[7]);
  }
  else
  {
    trackerpod_error("regenerate_eoc failed (rc = %d)!\n", rc);
    pdev->num_urbs = old_urbcount;
  }

  trackerpod_debug("regenerate_eoc rc = %d", rc);

  return rc;
}

void trackerpod_generate_pkt(__u8 *buffer, __u8 axis, __u8 value)
{
  buffer[0] = (((value & 0xF0) >> 4) ^ 0x01) | axis;
  buffer[1] = value & 0x0F;

  buffer[6] = buffer[4] = buffer[2] = buffer[0];
  buffer[7] = buffer[5] = buffer[3] = buffer[1];
}

int trackerpod_move(struct trackerpod_device *pdev, int pan, int tilt)
{
  int rc = 0;
  __u8 buffer[8];

  trackerpod_debug("trackerpod_move(%02x, %02x) called", pan, tilt);

  pan  = CLAMP(pan,  TRACKERPOD_MIN_PAN,  TRACKERPOD_MAX_PAN);
  tilt = CLAMP(tilt, TRACKERPOD_MIN_TILT, TRACKERPOD_MAX_TILT);

  trackerpod_debug_pkt("trackerpod_move, clamped values = %02x, %02x",
		       pan, tilt);

  trackerpod_generate_pkt(buffer, TRACKERPOD_AXIS_PAN, pan);
  rc = (rc != 0) ? rc : trackerpod_send_multiple(pdev, buffer, 2);

  trackerpod_generate_pkt(buffer, TRACKERPOD_AXIS_TILT, tilt);
  rc = (rc != 0) ? rc : trackerpod_send_multiple(pdev, buffer, 2);

  rc = (rc != 0) ? rc : trackerpod_send_multiple(pdev, NULL, 2);

  if (rc == 0)
  {
    pdev->pan  = pan;
    pdev->tilt = tilt;
  }

  trackerpod_debug("trackerpod_move rc = %d", rc);

  return rc;
}


int trackerpod_initialise(struct trackerpod_device *pdev)
{
  int rc = 0;

  trackerpod_debug("trackerpod_initialise called");

  if (pdev->initialised)
  {
    trackerpod_warn("Already initialised");
    return 0;
  }

  if (pdev->unplugged)
  {
    trackerpod_warn("Cannot initialise unplugged device");
    return 0;
  }

  rc = (rc != 0) ? rc : trackerpod_send_multiple(pdev,
		 		 "\x63\xc2\xe3\x22\xe3\x22\xe3\x22", 5);
  rc = (rc != 0) ? rc : trackerpod_recv(pdev);
  rc = (rc != 0) ? rc : trackerpod_send_multiple(pdev,
		 		 "\x01\x03\x01\x03\x01\x03\x01\x03", 2);
  rc = (rc != 0) ? rc : trackerpod_send_multiple(pdev,
		 		 "\x01\x00\x01\x00\x01\x00\x01\x00", 2);
  rc = (rc != 0) ? rc : trackerpod_send_multiple(pdev,
		 		 "\x21\x00\x21\x00\x21\x00\x21\x00", 2);
  rc = (rc != 0) ? rc : trackerpod_move(pdev, 0x80, 0x80);

  if (!rc)
  {
    pdev->initialised = 1;
  }

  trackerpod_debug("trackerpod_initialise rc = %d", rc);

  return rc;
}

int trackerpod_shutdown(struct trackerpod_device *pdev)
{
  int rc = 0;

  trackerpod_debug("trackerpod_shutdown called");

  if (!pdev->initialised)
  {
    trackerpod_warn("Already shutdown");
    return 0;
  }

  if (pdev->unplugged)
  {
    trackerpod_warn("Cannot shutdown unplugged device");
    return 0;
  }

  rc = (rc != 0) ? rc : trackerpod_send_multiple(pdev,
		 		 "\x71\x00\x71\x00\x71\x00\x71\x00", 2);
  rc = (rc != 0) ? rc : trackerpod_send_multiple(pdev,
		 		 "\x21\x00\x21\x00\x21\x00\x21\x00", 2);
  rc = (rc != 0) ? rc : trackerpod_send_multiple(pdev,
		 		 "\x1e\x0f\x1e\x0f\x1e\x0f\x1e\x0f", 2);

  if (!rc)
  {
    pdev->initialised = 0;
  }

  trackerpod_debug("trackerpod_shutdown rc = %d", rc);

  return rc;
}

static struct trackerpod_device **minor_to_pdev;

void *trackerpod_probe(struct usb_device *udev, unsigned int interface,
		const struct usb_device_id *id)
{
  struct trackerpod_device *pdev;
  int minor = 0;

  while (minor_to_pdev[minor] != NULL && ++minor < TRACKERPOD_MAX_DEVS);

  if (minor >= TRACKERPOD_MAX_DEVS)
  {
    printk(KERN_WARNING TRACKERPOD_NAME ": Too many devices\n");
    return NULL;
  }

  if ((pdev = kmalloc(sizeof(struct trackerpod_device), GFP_KERNEL)) == NULL)
  {
    printk(KERN_WARNING TRACKERPOD_NAME \
		    ": Couldn't kmalloc device structure!\n");
    return NULL;
  }

  minor_to_pdev[minor] = pdev;

  trackerpod_populate_struct(pdev, minor);
  pdev->udev = udev;

  printk(KERN_INFO TRACKERPOD_NAME \
		  ": Registered new device %s, using minor %d\n",
		  pdev->name, TRACKERPOD_MINOR + pdev->minor);

  return pdev;
}

void trackerpod_disconnect(struct usb_device *dev, void *ptr)
{
  struct trackerpod_device *pdev = (struct trackerpod_device *) ptr;

  if (pdev == NULL)
  {
    printk(KERN_ERR TRACKERPOD_NAME \
		    ": trackerpod_disconnect called with NULL pdev!\n");
    return;
  }

  trackerpod_debug("trackerpod_disconnect called");

  pdev->unplugged = 1;

  if (pdev->open)
  {
    trackerpod_debug("trackerpod_disconnect called with open devices");
    // TODO - check this
    sleep_on(&pdev->remove_ok);
  }

  // TODO - this is broken. Only do this at module unload time...
  if (pdev->initialised)
  {
    trackerpod_shutdown(pdev);
  }

  printk(KERN_INFO TRACKERPOD_NAME ": De-registered %s\n", pdev->name);

  minor_to_pdev[pdev->minor] = NULL;
  kfree(pdev);
}

static int trackerpod_open(struct inode *inode, struct file *file)
{
  struct trackerpod_device *pdev;
  int rc;

  if (   MINOR(inode->i_rdev) < TRACKERPOD_MINOR
      || MINOR(inode->i_rdev) >= (TRACKERPOD_MINOR + TRACKERPOD_MAX_DEVS))
  {
    printk(KERN_ERR TRACKERPOD_NAME \
		    ": trackerpod_open called for invalid minor: %d\n",
		    MINOR(inode->i_rdev));
    return -EIO;
  }

  pdev = minor_to_pdev[MINOR(inode->i_rdev) - TRACKERPOD_MINOR];

  if (pdev == NULL || pdev->unplugged)
  {
    return -ENODEV;
  }

  trackerpod_debug("trackerpod_open called");

  down(&pdev->lock);

  if (pdev->open)
  {
    trackerpod_msg("trackerpod_open - device opened multiple times");
    /*
    up(&pdev->lock);
    return -EBUSY;
    */
  }

  if (!pdev->initialised && (rc = trackerpod_initialise(pdev)) < 0)
  {
    trackerpod_error("trackerpod_initialise failed");
    up(&pdev->lock);
    return rc;
  }

  pdev->open++;
  MOD_INC_USE_COUNT;

  file->private_data = pdev;

  up(&pdev->lock);

  trackerpod_debug("trackerpod_open rc = 0");

  return 0;
}

static int trackerpod_ioctl(struct inode *inode, struct file *file,
		unsigned int cmd, unsigned long arg)
{
  int rc = 0;
  struct trackerpod_posn p;
  struct trackerpod_device *pdev;

  if ((pdev = file->private_data) == NULL)
  {
    printk(KERN_ERR TRACKERPOD_NAME\
		    ": private_data NULL in trackerpod_ioctl\n");
    return -EINVAL;
  }

  trackerpod_debug("trackerpod_ioctl called");

  if (_IOC_TYPE(cmd) != TRACKERPOD_IOC_MAGIC)
  {
    trackerpod_debug("Wrong magic on ioctl");
    return -EINVAL;
  }

  switch (cmd)
  {
    case TRACKERPOD_IOCSPOSN:
      if (copy_from_user(&p, (void *)arg, sizeof(struct trackerpod_posn)))
      {
        trackerpod_debug("TRACKERPOD_IOCSPOSN: copy_from_user failed");
        rc = -EFAULT;
      }
      else
      {
        down(&pdev->lock);
        rc = trackerpod_move(pdev, p.pan, p.tilt);
        up(&pdev->lock);
      }
      break;

    case TRACKERPOD_IOCGPOSN:
      p.pan  = pdev->pan;
      p.tilt = pdev->tilt;

      if (copy_to_user((void *)arg, &p, sizeof(struct trackerpod_posn)))
      {
        trackerpod_debug("TRACKERPOD_IOCGPOSN: copy_to_user failed");
	rc = -EFAULT;
      }
      break;

#if 0
    case TRACKERPOD_IOCSHCL:
      pdev->shutdown_on_close = 1;
      break;

    case TRACKERPOD_IOCNSHCL:
      pdev->shutdown_on_close = 0;
      break;
#endif

    default:
      trackerpod_debug("Unsupported ioctl");
      rc= -EINVAL;
  }

  trackerpod_debug("trackerpod_ioctl rc = %d", rc);

  return rc;
}

static int trackerpod_release(struct inode *inode, struct file *file)
{
  struct trackerpod_device *pdev;

  if (   MINOR(inode->i_rdev) < TRACKERPOD_MINOR
      || MINOR(inode->i_rdev) >= (TRACKERPOD_MINOR + TRACKERPOD_MAX_DEVS))
  {
    printk(KERN_ERR TRACKERPOD_NAME \
		    ": trackerpod_close called for invalid minor: %d\n",
		    MINOR(inode->i_rdev));
    return -EIO;
  }

  if ((pdev = file->private_data) == NULL)
  {
    printk(KERN_WARNING TRACKERPOD_NAME \
		    ": private_data NULL in trackerpod_release!\n");
    pdev = minor_to_pdev[MINOR(inode->i_rdev) - TRACKERPOD_MINOR];
  }

  trackerpod_debug("trackerpod_release called");

  if (pdev == NULL)
  {
    return -ENODEV;
  }

  down(&pdev->lock);

  if (!pdev->open)
  {
    trackerpod_error("trackerpod_release called when device not open!");
    up(&pdev->lock);
    return -EIO;
  }

  if (pdev->shutdown_on_close && trackerpod_shutdown(pdev) < 0)
  {
    trackerpod_error("trackerpod_shutdown failed");
  }

  file->private_data = NULL;

  pdev->open--;
  MOD_DEC_USE_COUNT;

  up(&pdev->lock);

  if (pdev->unplugged && !pdev->open)
  {
    wake_up(&pdev->remove_ok);
  }

  trackerpod_debug("trackerpod_release rc = 0");

  return 0;
}

static struct file_operations trackerpod_file_ops =
{
  owner:	THIS_MODULE,
  open:		trackerpod_open,
  ioctl:	trackerpod_ioctl,
  release:	trackerpod_release,
};

static struct usb_driver trackerpod_usb_ops =
{
  name:		"trackerpod",
  probe:	trackerpod_probe,
  disconnect:	trackerpod_disconnect,
  id_table:	trackerpod_id_table,
  fops:		&trackerpod_file_ops,
  minor:	TRACKERPOD_MINOR,
};

int init_module(void)
{
  int rc;

  minor_to_pdev = kmalloc(sizeof(struct trackpod_device *)
		          * TRACKERPOD_MAX_DEVS, GFP_KERNEL);

  if (minor_to_pdev == NULL)
  {
    printk(KERN_WARNING TRACKERPOD_NAME \
		  ": kmalloc for minor_to_pdev failed\n");
    return -ENOMEM;
  }

  memset(minor_to_pdev, 0,
		  sizeof(struct trackpod_device *) * TRACKERPOD_MAX_DEVS);

  if ((rc = usb_register(&trackerpod_usb_ops)) < 0)
  {
    printk(KERN_WARNING TRACKERPOD_NAME ": usb_register failed\n");
    kfree(minor_to_pdev);
    return rc;
  }

  return 0;
}

void cleanup_module(void)
{
  usb_deregister(&trackerpod_usb_ops);
  kfree(minor_to_pdev);
}
