
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>

#include <linux/version.h>

#include "in.ipscrompd.h"

#if LINUX_VERSION_CODE >= 131430 /* 2.1.102 */
#define NEW_FIREWALL
#endif

#ifdef NEW_FIREWALL

/* There are two different ways of specifying the chain we append   */
/* to. Pick one of them.                                            */
/* Also remember that the maximum length of a chain name is 8 chars */

/* First way, hardcoded 'CHAIN':                                    */
/* #define CHAIN	"ipscromp" */
/* #define CHAIN	IP_FW_LABEL_INPUT */

/* Second way, dynamic chain:                                       */
/* Rotate a number of different chains based on the hour of the     */
/* day. Chain number = hour / (24 / DYNAMIC_CHAIN_COUNT) so pick a  */
/* value that 24 divides into unless you want different times for   */
/* some chains.                                                     */

#define CHAIN		"scromp-%d"
#define DYNAMIC_CHAIN_COUNT	3

#endif

/* With glibc2, various linux/ header files do not get on with their */
/* sys/ equivalents. Unfortunately, we need several linux/ headers   */
/* for the firewall stuff. Thus, what follows is something of a      */
/* fudge. Any better suggestions would be appreciated.               */

#if defined(__GLIBC__) && __GLIBC__ >= 2 && __GLIBC_MINOR__ < 1
/* glibc 2.0 - broken headers */

#include <linux/in.h>
#include <linux/inet.h>
#include <linux/socket.h>

/* These are to quiet gcc warning messages, since they are only      */
/* defined in the sys/ headers                                       */
int socket(int domain, int type, int protocol);
int setsockopt(int s, int level, int optname,
               const void *optval, int optlen);

#else /* Sensible LIBC */

/* This smooths out the only wrinkle with glibc2.1                   */
#define _LINUX_BYTEORDER_GENERIC_H

#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/socket.h>

#endif

/* And yes, all these headers /really are/ needed.                   */
#include <linux/inet.h>
#include <linux/ip.h>
#include <linux/tcp.h>
#include <linux/udp.h>
#include <linux/icmp.h>
#include <linux/if.h>
#include <linux/ip_fw.h>

/* Conflicts with some stuff in linux/icmp.h if included first       */
#include <errno.h>

int do_setsockopt(int cmd, void *data, size_t datalen)
{
  static int sockfd = -1;

  if (sockfd < 0)
  {
    sockfd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
    if (sockfd < 0)
    {
      return -errno;
    }
  }

  if (setsockopt(sockfd, IPPROTO_IP, cmd, (char *)data, datalen) < 0)
  {
    return -errno;
  }

  return 0;
}


int fw_add_ip(struct in_addr addr, char *user)
{
#ifdef NEW_FIREWALL
  int rc;
  struct ip_fwchange fwc;

#ifdef DYNAMIC_CHAIN_COUNT
  struct tm *time_now;
  time_t num_time = time(NULL);
#endif

  /* Empty the structure */
  memset(&fwc, 0, sizeof(fwc));

  /* Select chain we are appending to */
#ifdef DYNAMIC_CHAIN_COUNT

  time_now = localtime(&num_time);
  snprintf(fwc.fwc_label, sizeof(fwc.fwc_label),
           CHAIN, (time_now->tm_hour * DYNAMIC_CHAIN_COUNT) / 24);

#else
  memcpy(fwc.fwc_label, CHAIN, sizeof(CHAIN));
#endif

  /* The chain we want to jump to (ACCEPT) */
  memcpy(fwc.fwc_rule.label, IP_FW_LABEL_ACCEPT, sizeof(IP_FW_LABEL_ACCEPT));

  /* Source address (supplied) & mask */
  fwc.fwc_rule.ipfw.fw_src         = addr;
  fwc.fwc_rule.ipfw.fw_smsk.s_addr = 0xFFFFFFFF;

  /* Destination address (anywhere - we are on the INPUT chain) */
  fwc.fwc_rule.ipfw.fw_dst.s_addr  = 0x00000000;
  fwc.fwc_rule.ipfw.fw_dmsk.s_addr = 0x00000000;

  /* Source & dest port ranges (any) */
  fwc.fwc_rule.ipfw.fw_spts[0] = 0;
  fwc.fwc_rule.ipfw.fw_spts[1] = 0xFFFF;

  fwc.fwc_rule.ipfw.fw_dpts[0] = 0;
  fwc.fwc_rule.ipfw.fw_dpts[1] = 0xFFFF;

  /* TOS mangling (Do none) */
  fwc.fwc_rule.ipfw.fw_tosand = 0xFF;
  fwc.fwc_rule.ipfw.fw_tosxor = 0x00;

#ifdef DYNAMIC_CHAIN_COUNT
  
  rc = do_setsockopt(IP_FW_APPEND, &fwc, sizeof(fwc));
  if (rc >= 0)
  {
    rc = 24 / DYNAMIC_CHAIN_COUNT;
  }
  return rc;
#else
  return do_setsockopt(IP_FW_APPEND, &fwc, sizeof(fwc));  
#endif

#else /* notdef NEW_FIREWALL */

  struct ip_fw fw;

  /* Empty the structure */
  memset(&fw, 0, sizeof(fw));

  /* Source address (supplied) & mask */
  fw.fw_src         = addr;
  fw.fw_smsk.s_addr = 0xFFFFFFFF;

  /* Destination address (anywhere - we are on the INPUT chain) */
  fw.fw_dst.s_addr  = 0x00000000;
  fw.fw_dmsk.s_addr = 0x00000000;

  /* TOS mangling (Do none) */
  fw.fw_tosand = 0xFF;
  fw.fw_tosxor = 0x00;

  fw.fw_flg  = IP_FW_F_ALL | IP_FW_F_ACCEPT;

  return do_setsockopt(IP_FW_APPEND_IN, &fw, sizeof(fw));  
#endif /* NEW_FIREWALL */
}
